#! /usr/bin/env python3
#
# Script per il salvataggio dei dati di stampa di CUPS su DB SQLite
#
# Copyright (C) 2025-2026 FUSS Project <info@fuss.bz.it>
# Authors: Claudio Cavalli <ccavalli@fuss.bz.it>
#          Simone Piccardi <piccardi@truelite.it>
#          Elena Grandi  <elena@truelite.it>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as
# published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.
#

import datetime
import glob
import gzip
import os.path
import sqlite3


DB_PATH = "/var/log/cups/cupspages.db"


def insert_lines(f, cur):
    for line in f:
        line = line.strip()
        if line.startswith('"') and line.endswith('"'):
            line = line[1:-1]

        parts = line.split(",")

        if len(parts) < 6:
            print("Ignoring invalid line {}".format(line))
            continue

        printer = parts[0]
        user    = parts[1]
        pages   = parts[5]
        file    = parts[7]
        time    = parts[2][1:-1]
        try:
            date    = datetime.datetime.strptime(time, "%d/%b/%Y:%H:%M:%S %z")
        except ValueError:
            print("Ignoring invalid line {}".format(line))
            continue

        cur.execute("INSERT OR IGNORE INTO jobs VALUES (?, ?, ?, ?, ?)",
                    (user, pages, printer, file, date))


def main():
    # If the DB already exists we only want to read the latest log file,
    # otherwise we want to read everything
    log_files = []
    if os.path.exists(DB_PATH):
        log_files = ["/var/log/cups/page_log"]
    else:
        # pattern of log files
        pattern = '/var/log/cups/page_log*'
        log_files = glob.glob(pattern)

    # open DB file, and create table if absent
    con = sqlite3.connect(DB_PATH)
    cur = con.cursor()
    cur.execute("CREATE TABLE IF NOT EXISTS jobs (user TEXT, pages INTEGER, printer TEXT, file TEXT, time TEXT, PRIMARY KEY (user, time))")

    try:
        for filename in log_files:
            if filename.endswith('.gz'):
                f = gzip.open(filename, 'rt')
            else:
                f = open(filename, 'r')
            insert_lines(f, cur)
            con.commit()
    except FileNotFoundError:
        print("The file", filename, "was not found.")

    # closing DB connection
    con.close()


if __name__ == "__main__":
    main()
